//
//  PPActivityIndicatorView.m
//
//  Note
//  -----------------------
//  1. 使用PPAIVRotateLayer的目的是不要旋轉整個view，但繪圖的方式還是統一在PPActivityIndicatorView中處理。
//  2. Apply custom appearance need todo:
//     - property宣告後加上 UI_APPEARANCE_SELECTOR
//     - property必須是標準的iOS type
//     - 覆寫+(void)initialize，加入appearance預設值。
//     - 實作property的setter
//

#import "PPActivityIndicatorView.h"
#import "UIColor+BlendColor.h"

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Const define

NSString * const PPAIV_AnimationKey = @"PPAIV_AnimationKey";

// GradientCircleStyle的預設線寬
CGFloat const PPAIV_GradientCircleStyleDefaultLineWidth = 2.0;

// GradientCircleStyle的分割數
NSInteger const PPAIV_GradientCircleStyleTotalDivision = 100;



// SystemStyle的(內緣半徑/外緣半徑)比例
CGFloat const PPAIV_SystemStyleInnerRadiusRatio = 0.7;

// SystemStyle的預設線寬
CGFloat const PPAIV_SystemStyleDefaultLineWidth = 2.0;

// SystemStyle的分割數
NSInteger const PPAIV_SystemStyleTotalDivision = 12;




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Inline define

inline static dispatch_source_t CreateDispatchTimer(float interval, dispatch_queue_t queue, dispatch_block_t block)
{
    dispatch_source_t timer = dispatch_source_create(DISPATCH_SOURCE_TYPE_TIMER, 0, 0, queue);
    if (timer)
    {
        // GCD use nano second
        dispatch_source_set_timer(timer, dispatch_walltime(NULL, 0), interval*1000000000, 0);
        dispatch_source_set_event_handler(timer, block);
        dispatch_resume(timer);
    }
    return timer;
}

inline static void DestroyDispatchTimer(dispatch_source_t timer)
{
    if (timer)
    {
        dispatch_source_cancel(timer);
        dispatch_release(timer);
    }
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPAIVRotateLayer
@class PPAIVRotateLayer;
@protocol PPAIVRotateLayerDrawDelegate <NSObject>
- (void)rotateLayer:(PPAIVRotateLayer *)rotateLayer drawInContext:(CGContextRef)context;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface PPAIVRotateLayer : CALayer
@property (nonatomic, assign) id<PPAIVRotateLayerDrawDelegate> drawDelegate;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPAIVRotateLayer

//================================================================================
//
//================================================================================
- (void)drawInContext:(CGContextRef)ctx
{
    [self.drawDelegate rotateLayer:self drawInContext:ctx];
}

@end





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPActivityIndicatorView interface

@interface PPActivityIndicatorView () <PPAIVRotateLayerDrawDelegate>

@property (nonatomic, retain) NSNumber *visibleHeight;
@property (nonatomic, retain) PPAIVRotateLayer *rotateLayer;

@property (nonatomic, assign) dispatch_source_t timer;
@property (nonatomic, readwrite) CGFloat currentProgress;
@property (nonatomic, readwrite) NSInteger totalDivision;
@property (nonatomic, readwrite) CGFloat lineWidth;
@property (atomic, assign) BOOL isIndicatorAnimating;
@property (atomic, assign) BOOL needAnimating;

@end




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPActivityIndicatorView implementation

@implementation PPActivityIndicatorView

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//================================================================================
// Appearance預設值
//================================================================================
+ (void)initialize
{
//    NSLog(@"%s %d", __func__, (int)PPActivityIndicatorViewStyleGradientCircle);
    
    [[PPActivityIndicatorView appearance] setStyle:PPActivityIndicatorViewStyleGradientCircle];
    [[PPActivityIndicatorView appearance] setLineColor:[UIColor whiteColor]];
    [[PPActivityIndicatorView appearance] setBackgroundColor:[UIColor clearColor]];
}


//================================================================================
//
//================================================================================
- (instancetype)init
{
//    NSLog(@"%s", __func__);

    if(self = [super init])
    {
        self.style = [[PPActivityIndicatorView appearance] style];
        self.lineColor = [[PPActivityIndicatorView appearance] lineColor];
        self.backgroundColor = [[PPActivityIndicatorView appearance] backgroundColor];
        
        [self addRotateLayer];
        
        [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(recvDidEnterBackground:) name:UIApplicationDidEnterBackgroundNotification object:nil];
        [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(recvWillEnterForeground:) name:UIApplicationWillEnterForegroundNotification object:nil];
        
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (instancetype)initWithFrame:(CGRect)frame
{
    if(self = [super initWithFrame:frame])
    {
        [self addRotateLayer];
        
        [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(recvDidEnterBackground:) name:UIApplicationDidEnterBackgroundNotification object:nil];
        [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(recvWillEnterForeground:) name:UIApplicationWillEnterForegroundNotification object:nil];
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    
    self.visibleHeight = nil;
    self.lineColor = nil;
    
    DestroyDispatchTimer(self.timer);
    self.timer = nil;

    [self.rotateLayer removeFromSuperlayer];
    self.rotateLayer = nil;
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Property methods

//================================================================================
//
//================================================================================
- (void)setStyle:(PPActivityIndicatorViewStyle)style
{
    if(_style != style)
    {
        _style = style;
     
        //////////////////////////////////////////////////
        // 各style顯示預設值
        
        switch (_style)
        {
            case PPActivityIndicatorViewStyleSystem:
            {
                self.totalDivision = PPAIV_SystemStyleTotalDivision;
                self.lineWidth = PPAIV_SystemStyleDefaultLineWidth;
                break;
            }
                
            case PPActivityIndicatorViewStyleGradientCircle:
            {
                self.totalDivision = PPAIV_GradientCircleStyleTotalDivision;
                self.lineWidth = PPAIV_GradientCircleStyleDefaultLineWidth;
                break;
            }
                
            default:
            {
                break;
            }
        }

        //////////////////////////////////////////////////

        [self setNeedsLayout];
        
        //////////////////////////////////////////////////
        
        if(self.needAnimating == YES)
        {
            [self startAnimating];
            self.needAnimating = NO;
        }
    }
}


//================================================================================
//
//================================================================================
- (void)setLineColor:(UIColor *)lineColor
{
    if(_lineColor != lineColor)
    {
        [_lineColor release];
        _lineColor = [lineColor retain];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Override methods

//================================================================================
//
//================================================================================
- (void)layoutSubviews
{
//    NSLog(@"%s %@", __func__, NSStringFromCGRect(self.bounds));
    
    [super layoutSubviews];
    
    //////////////////////////////////////////////////
    
    self.rotateLayer.frame = self.bounds;
    [self.rotateLayer setNeedsDisplay];
}


//================================================================================
//
//================================================================================
- (void)drawRect:(CGRect)rect
{
//    NSLog(@"%s", __func__);
    
    [super drawRect:rect];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Notification receiver

//==============================================================================
//
//==============================================================================
- (void)recvDidEnterBackground:(NSNotification *)notification
{
    if(self.isIndicatorAnimating == YES)
    {
        [self stopAnimating];
        self.needAnimating = YES;
    }
}


//==============================================================================
//
//==============================================================================
- (void)recvWillEnterForeground:(NSNotification *)notification
{
    if(self.needAnimating == YES)
    {
        [self startAnimating];
        self.needAnimating = NO;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPAIVRotateLayerDrawDelegate methods

//================================================================================
//
//================================================================================
- (void)rotateLayer:(PPAIVRotateLayer *)rotateLayer drawInContext:(CGContextRef)context
{
//    NSLog(@"%s", __func__);

    switch (self.style)
    {
        case PPActivityIndicatorViewStyleSystem:
        {
            [self drawSystemStyleIndicatorWithContext:context];
            break;
        }
            
        case PPActivityIndicatorViewStyleGradientCircle:
        {
            @autoreleasepool
            {
                // !! 先畫2倍大的image再縮到context上，鋸齒才不會明顯。
                
                CGFloat scale = 2.0;
                CGFloat imageLineWidth = self.lineWidth * scale;
                CGFloat imageVisibleHeight = self.visibleHeight ? [self.visibleHeight floatValue] * scale : 9999;
                CGSize  imageSize = CGSizeMake(rotateLayer.bounds.size.width * scale,
                                               rotateLayer.bounds.size.height * scale);
                
                UIGraphicsBeginImageContextWithOptions(imageSize, NO, 1.0);
                
                [self drawGradientCircleStyleIndicatorWithSize:imageSize
                                                     lineWidth:imageLineWidth
                                                 visibleHeight:imageVisibleHeight];
                
                UIImage *image = UIGraphicsGetImageFromCurrentImageContext();
                UIGraphicsEndImageContext();
                
                //////////////////////////////////////////////////
                
                UIGraphicsPushContext(context);
                [image drawInRect:rotateLayer.bounds];
                UIGraphicsPopContext();
            }
            
            break;
        }
            
        default:
        {
            break;
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods

//================================================================================
//
//================================================================================
- (void)addRotateLayer
{
    self.rotateLayer = [[[PPAIVRotateLayer alloc] init] autorelease];
    
    if(self.rotateLayer != nil)
    {
        self.rotateLayer.drawDelegate = self;
        [self.layer insertSublayer:self.rotateLayer atIndex:0];
    }
}


//================================================================================
//
//================================================================================
- (void)drawSystemStyleIndicatorWithContext:(CGContextRef)context
{
    CGContextSetLineCap(context, kCGLineCapRound);
    CGContextSetLineWidth(context, self.lineWidth);
    
    //////////////////////////////////////////////////
    
    CGFloat outerRadius = ceil(fminf(self.bounds.size.width, self.bounds.size.height)/3.0);
    CGFloat innerRadius = ceil(outerRadius * PPAIV_SystemStyleInnerRadiusRatio);
    CGPoint beginPoint = CGPointMake(0, innerRadius);
    CGPoint endPoint = CGPointMake(0, outerRadius);
    
    CGPoint boundsCenter = CGPointMake(CGRectGetMidX(self.bounds), CGRectGetMidY(self.bounds));
    CGFloat anglePerDivision = -(360/self.totalDivision)*M_PI/180;
    CGAffineTransform t = CGAffineTransformIdentity;
    
    t = CGAffineTransformTranslate(t, boundsCenter.x, boundsCenter.y);
    t = CGAffineTransformScale(t, 1, -1);

    if(self.isAnimating == NO) // 下拉過程顯示 (只顯示部份的圓形範圍，由淺到深)
    {
        int endDivision = (int)(self.currentProgress * self.totalDivision);
        
        for(int i=0; i<=endDivision; i++)
        {
            CGPoint beginPoint1 = CGPointApplyAffineTransform(beginPoint, t);
            CGPoint endPoint1 = CGPointApplyAffineTransform(endPoint, t);
            
            if(self.visibleHeight != nil &&
               fmaxf(beginPoint1.y, endPoint1.y) > [self.visibleHeight floatValue])
            {
                break;
            }
            
            UIColor *strokeColor = [self.lineColor colorWithAlphaComponent:(CGFloat)i/(CGFloat)endDivision];
            CGContextSetStrokeColorWithColor(context, strokeColor.CGColor);
            CGContextMoveToPoint(context, beginPoint1.x, beginPoint1.y);
            CGContextAddLineToPoint(context, endPoint1.x, endPoint1.y);
            
            t = CGAffineTransformRotate(t, anglePerDivision);
            CGContextStrokePath(context);
        }
    }
    else // 動畫顯示 (顯示完整的圓形範圍，currentProgress顏色最深，左右遞減。)
    {
        int left = self.currentProgress-1;
        
        if(left < 0)
        {
            left += self.totalDivision;
        }
        
        int right = self.currentProgress+1;
        
        if(right > self.totalDivision)
        {
            right -= self.totalDivision;
        }
        
        for(int i=0; i<self.totalDivision; i++)
        {
            CGPoint beginPoint1 = CGPointApplyAffineTransform(beginPoint, t);
            CGPoint endPoint1 = CGPointApplyAffineTransform(endPoint, t);
            
            if(self.visibleHeight != nil &&
               fmaxf(beginPoint1.y, endPoint1.y) > [self.visibleHeight floatValue])
            {
                break;
            }
            
            UIColor *strokeColor = nil;
            
            if(i == self.currentProgress)
            {
                strokeColor = self.lineColor;
            }
            else if(i==left || i==right)
            {
                strokeColor = [self.lineColor colorWithAlphaComponent:0.7];
            }
            else
            {
                strokeColor = [self.lineColor colorWithAlphaComponent:0.2];
            }
            
            CGContextSetStrokeColorWithColor(context, strokeColor.CGColor);
            CGContextMoveToPoint(context, beginPoint1.x, beginPoint1.y);
            CGContextAddLineToPoint(context, endPoint1.x, endPoint1.y);
            
            t = CGAffineTransformRotate(t, anglePerDivision);
            CGContextStrokePath(context);
        }
    }
}


//================================================================================
//
//================================================================================
- (void)drawGradientCircleStyleIndicatorWithSize:(CGSize)size
                                       lineWidth:(CGFloat)lineWidth
                                   visibleHeight:(CGFloat)visibleHeight
{
    CGFloat currentAngle = 0.0;
    CGRect circleRect = CGRectMake(lineWidth,
                                   lineWidth,
                                   size.width - 2*lineWidth,
                                   size.height - 2*lineWidth);
    
    CGPoint center = CGPointMake(CGRectGetMidX(circleRect), CGRectGetMidY(circleRect));
    CGFloat radius = ceil(fminf(circleRect.size.width/2.0, circleRect.size.height/2.0));
    
    // !! 若要從90度開始需要偏移一度，不然繪圖時會有問題。
    int beginDegree = 0;
    CGFloat beginAngle = -(CGFloat)M_PI_2 + ((CGFloat)beginDegree/360.0*2.0*(CGFloat)M_PI);
    
    for(float i=0; i<self.currentProgress; i+=0.01)
    {
        @autoreleasepool
        {
            CGFloat endAngle = i*2.0*(CGFloat)M_PI + beginAngle;
            
            if(currentAngle == 0.0)
            {
                currentAngle = beginAngle;
            }
            else
            {
                currentAngle = endAngle - 0.1;
            }
            
            
            UIBezierPath *arcPath = [UIBezierPath bezierPathWithArcCenter:center
                                                                   radius:radius
                                                               startAngle:currentAngle
                                                                 endAngle:endAngle
                                                                clockwise:YES];
            
            //////////////////////////////////////////////////
            // !! 沒有高度不需要畫，計算下一個。
            
            if(arcPath.bounds.size.height == 0)
            {
                continue;
            }
            
            
            //////////////////////////////////////////////////
            // !! 有可見高度限制時，超過可見高度，後面的就不用畫了。
            //    (只有updateProgress時需要檢查)
            
//            NSLog(@"arcHeight %f", arcPath.bounds.origin.y+arcPath.bounds.size.height);
//            NSLog(@"visibleHeight %f", visibleHeight);
            
            if(arcPath.bounds.origin.y+arcPath.bounds.size.height > visibleHeight)
            {
                break;
            }
            
            
            //////////////////////////////////////////////////
            
            UIColor *strokeColor = [[UIColor clearColor] colorWithBlendColor:self.lineColor ratio:i/self.currentProgress];
            
            [strokeColor setStroke];
            arcPath.lineWidth = lineWidth;
            arcPath.lineCapStyle = kCGLineCapRound;
            
            [arcPath stroke];
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance methods

//================================================================================
//
//================================================================================
- (void)startAnimating
{
    if(self.isIndicatorAnimating == YES)
    {
        return;
    }

    //////////////////////////////////////////////////
    
    switch (self.style)
    {
        case PPActivityIndicatorViewStyleGradientCircle: // 旋轉layer
        {
            self.visibleHeight = nil;
            self.currentProgress = 1.0;
            [self.rotateLayer setNeedsDisplay];
            
            CABasicAnimation *rotation;
            rotation = [CABasicAnimation animationWithKeyPath:@"transform.rotation"];
            rotation.fromValue = [NSNumber numberWithFloat:0];
            rotation.toValue = [NSNumber numberWithFloat:(2 * M_PI)];
            rotation.duration = 0.7f; // Speed
            rotation.repeatCount = HUGE_VALF; // Repeat forever. Can be a finite number.
            
            [self.rotateLayer removeAnimationForKey:PPAIV_AnimationKey];
            [self.rotateLayer addAnimation:rotation forKey:PPAIV_AnimationKey];
            
            self.isIndicatorAnimating = YES;
            break;
        }

        case PPActivityIndicatorViewStyleSystem: // 重繪
        {
            self.visibleHeight = nil;
            self.currentProgress = 0.0;

            //////////////////////////////////////////////////

            __block typeof(self) blockSelf = self;

            self.timer = CreateDispatchTimer(0.1, dispatch_get_main_queue(), ^{

                [blockSelf.rotateLayer setNeedsDisplay];
                blockSelf.currentProgress += 1.0;
                
                if(blockSelf.currentProgress >= self.totalDivision)
                {
                    blockSelf.currentProgress = 0.0;
                }
            });

            //////////////////////////////////////////////////

            self.isIndicatorAnimating = YES;
            break;
        }
            
        default:
        {
            self.isIndicatorAnimating = NO;
            self.needAnimating = YES;
            break;
        }
    }
}



//================================================================================
//
//================================================================================
- (void)stopAnimating
{
    if(self.isIndicatorAnimating == NO)
    {
        return;
    }
    
    //////////////////////////////////////////////////

    switch (self.style)
    {
        case PPActivityIndicatorViewStyleGradientCircle: // 旋轉layer
        {
            [self.rotateLayer removeAnimationForKey:PPAIV_AnimationKey];
            self.isIndicatorAnimating = NO;
            break;
        }
            
        case PPActivityIndicatorViewStyleSystem: // 重繪
        {
            DestroyDispatchTimer(self.timer);
            self.timer = nil;
            
            self.isIndicatorAnimating = NO;
            self.currentProgress = 0.0;
            break;
        }
            
        default:
        {
            self.isIndicatorAnimating = NO;
            break;
        }
    }
}


//================================================================================
//
//================================================================================
- (BOOL)isAnimating
{
    return self.isIndicatorAnimating;
}


//================================================================================
//
//================================================================================
- (void)updateCurrentProgress:(CGFloat)currentProgress withVisibleHeight:(NSNumber *)visibleHeight
{
//    NSLog(@"%s %f %f", __func__, currentProgress, visibleHeight);
    
    if (self.isIndicatorAnimating == YES)
    {
        return;
    }
    
    //////////////////////////////////////////////////
    
    if(self.currentProgress != currentProgress)
    {
        self.currentProgress = currentProgress;
        self.visibleHeight = visibleHeight;
        [self.rotateLayer setNeedsDisplay];
    }
}


@end
