//
//  PPSyncDataController.h
//  
//
//  Created by Mike Shih on 12/03/16.
//  Copyright (c) 2012年 Penpower. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "NSError+Custom.h"
#import "PPSyncCardModel.h"
#import "PPSyncGroupModel.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

// Denote that a function/method retains the nsobject that it returns.
// This fixes warnings generated by clang when we return a non-autoreleased
// object from a function.
// http://clang-analyzer.llvm.org/annotations.html#cocoa_mem
#ifndef NS_RETURNS_RETAINED
  #ifndef __has_feature  // Optional.
    #define __has_feature(x) 0  // Compatibility with non-clang compilers.
  #endif  // __has_feature

  #if __has_feature(attribute_ns_returns_retained)
    #define NS_RETURNS_RETAINED __attribute__((ns_returns_retained))
  #else
    #define NS_RETURNS_RETAINED
  #endif  // __has_feature(attribute_ns_returns_retained)
#endif

////////////////////////////////////////////////////////////////////////////////////////////////////

#define PPSyncVersion_Ignore 0

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPSyncDataControllerDelegate

@class PPSyncDataController;
@protocol PPSyncDataControllerDelegate <NSObject>
@optional

/**
 取得另一端對應的groupID

 @param syncDataController Instance of PPSyncDataController
 @param otherSideGroupID 來源的group ID
 @param error 取得錯誤資訊
 @return 對應的groupID
 */
- (NSString *)syncDataController:(PPSyncDataController *)syncDataController copyGroupIDWithOtherSideGroupID:(NSString *)otherSideGroupID error:(NSError **)error NS_RETURNS_RETAINED;

/**
 取得另一端對應的影像資料

 @param syncDataController Instance of PPSyncDataController
 @param imageType 影像類型
 @param error 取得錯誤資訊
 @return 對應的影像資料
 */
- (NSData *)syncDataController:(PPSyncDataController *)syncDataController copyOtherSideImageDataWithImageType:(PPSyncCardImageType)imageType error:(NSError **)error NS_RETURNS_RETAINED;

/**
 取得另一端對應的名片同步資料

 @param syncDataController Instance of PPSyncDataController
 @param uniqueID 來源的card ID
 @param fromVersion 名片資料的原來版本，不需轉換請填入PPSyncVersion_Ignore。
 @param toVersion 名片資料要轉換的版本，不需轉換請填入PPSyncVersion_Ignore。
 @param error 取得錯誤資訊
 @return 對應的同步資料
 */
- (PPSyncCardModel *)syncDataController:(PPSyncDataController *)syncDataController copyOtherSideSyncCardModelWithUniqueID:(NSString *)uniqueID fromVersion:(NSUInteger)fromVersion toVersion:(NSUInteger)toVersion error:(NSError **)error NS_RETURNS_RETAINED;

/**
 取得另一端對應的cardID

 @param syncDataController Instance of PPSyncDataController
 @param uniqueID 來源的card ID
 @param error 取得錯誤資訊
 @return 對應的card ID
 */
- (NSString *)syncDataController:(PPSyncDataController *)syncDataController copyOtherSideUniqueIDWithUniqueID:(NSString *)uniqueID error:(NSError **)error NS_RETURNS_RETAINED;

/**
 檢查錯誤是否代表名片不存在
 (GoogleContacts)

 @param syncDataController Instance of PPSyncDataController
 @param error 要檢查的
 @return YES代表名片不存在
 */
- (BOOL)syncDataController:(PPSyncDataController *)syncDataController isCardNotExistError:(NSError *)error;

/**
 變更名片包含的影像類型及ID
 (GoogleContacts)

 @param syncDataController Instance of PPSyncDataController
 @param imageDictionary 影像類型及ID
 @param uniqueID 來源的card ID
 @param error 取得錯誤資訊
 @return 成功或失敗
 */
- (BOOL)syncDataController:(PPSyncDataController *)syncDataController updateOtherSideImageDictionary:(NSMutableDictionary *)imageDictionary otherSideUniqueID:(NSString *)uniqueID error:(NSError **)error;

/**
 取得名片資料的進度
(GoogleContacts)
 
 @param syncDataController Instance of PPSyncDataController
 @param progress 0.0～1.0
 */
- (void)syncDataController:(PPSyncDataController *)syncDataController progressOfCopySyncCardModels:(CGFloat)progress;

/**
 強制調整SyncAction中的group modifiedTime
 (1)執行card相關動作時，若會導致group modifiedTime變更，請使用此函式調整。
 (2)在執行SyncAction時導致的時間異動，都應該修改SyncAction中的相關紀錄，避免下次同步時間比對不同產生新的SyncAction。
 (3)AddressBook / WorldCard 有此現象

 @param syncDataController Instance of PPSyncDataController
 @param dataDict Dictionary with {GroupID:ModifiedTime} mapping
 */
- (void)syncDataController:(id)syncDataController forceUpdateSyncActionGroupModifiedTimeWithDataDict:(NSDictionary *)dataDict;

/**
 強制調整SyncAction中的card modifiedTime
 (1)執行group相關動作時，若會導致card modifiedTime變更，請使用此函式調整。
 (2)在執行SyncAction時導致的時間異動，都應該修改SyncAction中的相關紀錄，避免下次同步時間比對不同產生新的SyncAction。
 (3)AddressBook 有此現象
 
 @param syncDataController Instance of PPSyncDataController
 @param dataDict Dictionary with {CardID:ModifiedTime} mapping
 */
- (void)syncDataController:(id)syncDataController forceUpdateSyncActionCardModifiedTimeWithDataDict:(NSDictionary *)dataDict;

/**
 下次同步時刪除指定名片
 
 @param syncDataController Instance of PPSyncDataController
 @param uniqueID 要刪除的cardID
 */
- (void)syncDataController:(id)syncDataController forceRemoveCardWhenNextSyncWithUniqueID:(NSString *)uniqueID;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

typedef NS_ENUM(NSUInteger, NSErrorPPSyncDataController_Code)
{
    NSErrorPPSyncDataController_Code_Network = NSErrorCustom_Code_UserDefine,
    NSErrorPPSyncDataController_Code_NullImage,
    NSErrorPPSyncDataController_Code_NoAnyFields
};

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPSyncDataController

@interface PPSyncDataController : NSObject
{
@protected
    id<PPSyncDataControllerDelegate>    delegate_;
    NSError                             *lastError_;
}


////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Property

@property(nonatomic,assign)             id<PPSyncDataControllerDelegate>    delegate;
@property(nonatomic,retain)             NSError                             *lastError;
@property(nonatomic,assign)             BOOL needSpecialCheck;  // 是否要做特定欄位檢查來強制更新
////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Methods





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark Group methods

/**
 * Copy groups afetr date
 *
 * @param afterDate for copy groups
 */
- (NSMutableArray *)copySyncGroupModelsAfterDate:(NSDate *)afterDate;

/**
 * Add group
 * (完成需填入uniqueID及lastModifiedDate)
 *
 * @param syncGroupModel to add
 * @return BOOL of result
 */
- (BOOL)addSyncGroupModel:(PPSyncGroupModel *)syncGroupModel;

/**
 * Update group
 *
 * @param syncGroupModel to update
 * @param fromVersion for update from version
 * @param toVersion for update to version
 * @return Result
 */
- (BOOL)updateSyncGroupModel:(PPSyncGroupModel *)syncGroupModel fromVersion:(NSUInteger)fromVersion toVersion:(NSUInteger)toVersion;

/**
 * Remove group
 *
 * @param uniqueID to remove group
 * @return BOOL of result
 */
- (BOOL)removeSyncGroupModelByUniqueID:(NSString *)uniqueID;





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark card methods

/**
 * Total card count
 *
 * @return Card count，nil表示沒取到值，請用lastError看錯誤。
 */
- (NSNumber *)totalSyncCardsCount;

/**
 * Check can sync
 *
 * @param uniqueID to sync card
 * @param lastModifiedDate for sync card
 * @return BOOL of result
 */
- (BOOL)canSyncCardWithUniqueID:(NSString *)uniqueID lastModifiedDate:(NSDate *)lastModifiedDate;

/**
 * Copy cards afetr date
 *
 * @param afterDate for copy cards
 */
- (NSMutableArray *)copySyncCardModelsAfterDate:(NSDate *)afterDate;

/**
 * Copy card
 *
 * @param uniqueID for card
 * @param fromVersion for copy from version
 * @param toVersion for copy to version
 * @return PPSyncCardModel of card
 */
- (PPSyncCardModel *)copySyncCardModelWithUniqueID:(NSString *)uniqueID fromVersion:(NSUInteger)fromVersion toVersion:(NSUInteger)toVersion;

/**
 * Check if error is card not exist
 *
 * @param error to check
 * @return BOOL of result
 */
- (BOOL)isCardNotExistError:(NSError *)error;

/**
 * Add card
 * (完成需填入uniqueID及lastModifiedDate)
 *
 * @param syncCardModel to add
 * @return BOOL of result
 */
- (BOOL)addSyncCardModel:(PPSyncCardModel *)syncCardModel;

/**
 * Update card
 * (完成需填入lastModifiedDate)
 *
 * @param syncCardModel to update
 * @param fromVersion for update from version
 * @param toVersion for update to version
 * @return BOOL of resultb
 */
- (BOOL)updateSyncCardModel:(PPSyncCardModel *)syncCardModel fromVersion:(NSUInteger)fromVersion toVersion:(NSUInteger)toVersion;

/**
 * Remove card
 *
 * @param uniqueID to remove card
 * @return BOOL of result
 */
- (BOOL)removeSyncCardModelByUniqueID:(NSString *)uniqueID;





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark image methods

/**
 * Update image
 * (syncCardModel 裡面的 |imageDictionary| 若有更新的話，就會送到這個)
 *
 * @return BOOL of result
 */
- (BOOL)updateImageDictionary:(NSMutableDictionary *)imageDictionary uniqueID:(NSString *)uniqueID;

/**
 * Copy image
 *
 * @param uniqueID for card
 * @param imageType for image type
 * @return NSData of image
 */
- (NSData *)copyImageDataByUniqueID:(NSString *)uniqueID imageType:(PPSyncCardImageType)imageType;





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark retry methods

/**
 * Should reschedule action
 *
 * @param error to check should reschedule action
 * @return BOOL of result
 */
- (BOOL)shouldRescheduleActionWithError:(NSError *)error;

/**
 * Should retry action
 *
 * @param error to check should retry action
 * @return BOOL of result
 */
- (BOOL)shouldRetryActionWithError:(NSError *)error;





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark other methods

/**
 * Cancel sync action
 * (async mode, 一定要可以把目前執行的method中止)
 *
 */
- (void)cancel;

@end
