//
//  PPLogController.h
//  
//
//  Created by Mike on 13/5/7.
//  Modified by Egg on 14/6/27
//
//  Copyright (c) 2013年 Penpower. All rights reserved.
//
// Model
#import <Foundation/Foundation.h>

// View

// Controller

////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef PPLogMacro
#define PPLogData(va_mask, va_name, va_data)    [PPLogController logWithMask:(PPLogControllerMask)va_mask dataName:va_name data:va_data]
#define PPLog(va_mask, va_args, ...)            [PPLogController logWithMask:(PPLogControllerMask)va_mask format:(va_args), ##__VA_ARGS__]
#define PPLogFunction(va_args, ...)             PPLog(PPLogControllerMask_Normal, (@"%s " va_args), __FUNCTION__, ##__VA_ARGS__)
#define PPLogFunctionIn(va_args, ...)           PPLogFunction(@"in " va_args, ##__VA_ARGS__)
#define PPLogFunctionOut(va_args, ...)          PPLogFunction(@"out " va_args, ##__VA_ARGS__)

#define PPLog_StartLogCostTime(va_logkey)            [PPLogController startLogCostTime:(va_logkey)]
#define PPLog_LogCostTime(va_logkey, va_mask, va_args, ...)            [PPLogController logWithKey:(va_logkey) mask:(PPLogControllerMask)va_mask format:(va_args), ##__VA_ARGS__]
#define PPLog_StopLogCostTime(va_logkey)            [PPLogController stopLogCostTime:(va_logkey)]

#else
#define PPLogData(...)
#define PPLog(...)
#define PPLogFunction(...)
#define PPLogFunctionIn(...)
#define PPLogFunctionOut(...)

#define PPLog_StartLogCostTime(...)
#define PPLog_LogCostTime(...)
#define PPLog_StopLogCostTime(...)

#endif

////////////////////////////////////////////////////////////////////////////////////////////////////

#define PPLogController_DateFormat   @"yy/MM/dd HH:mm:ss:SSS"
#define PPLogController_FileMaxSize  2*1024*1024	//2MB





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Enum

typedef NS_OPTIONS(NSUInteger, PPLogControllerMask)
{
	PPLogControllerMask_None    = 0,
	PPLogControllerMask_Normal  = 1<<0,
    PPLogControllerMask_Custom  = 1<<1,
	PPLogControllerMask_leak    = 1<<2,
};

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface PPLogController : NSObject
{
	NSString		*fileName_;
    BOOL            toConsole_;
    BOOL            toFile_;
    uint32_t        mask_;
    NSFileHandle	*fileHandle_;
    NSString		*filePath_;
    NSString		*atPath_;
    
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Property

@property (nonatomic,retain, readonly) NSString *fileName;
@property (nonatomic,retain, readonly) NSString *filePath;
@property (nonatomic,assign          ) BOOL     toConsole;
@property (nonatomic,assign          ) BOOL     toFile;
@property (nonatomic,assign          ) uint32_t mask;

/// 保留log數量，超過時從最舊的開始刪除。（0代表無限制）
@property (nonatomic,assign          ) NSUInteger remainCount;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Methods

/*!
 *  Write NsData to specific file name under self.filePath,
 *  you need to call setFileName:atPath: before calling this method.
 *
 *  Note: use PPLogData() macro instead of calling this method directly,
 *        unless you specifically intend to do so.
 *
 *  @param mask     Mask of log level.
 *  @param dataName Name of the file to save data with.
 *  @param data     The data to save.
 *
 *  @return Return YES if successfully write data to file.
 */
- (BOOL)logWithMask:(PPLogControllerMask)mask dataName:(NSString *)dataName data:(NSData *)data;

/*!
 *  Output log.
 *
 *  Note: use PPLog() macro instead of calling this method directly,
 *        unless you specifically intend to do so.
 *
 *  @param mask   Mask of log level.
 *  @param format Format of string.
 *
 *  @return Return YES if log message is successfully composed by format and arguments.
 *          The returned value doesn't represent whether or not log was output to console or file.
 */
- (BOOL)logWithMask:(PPLogControllerMask)mask format:(NSString *)format, ...;

/*!
 *  Output log with arguments.
 *
 *  Note: use PPLog() macro instead of calling this method directly,
 *        unless you specifically intend to do so.
 *
 *  @param mask      Mask of log level.
 *  @param format    Format of string.
 *  @param arguments One or more arguments to be inserted into the string.
 *
 *  @return Return YES if log message is successfully composed by format and arguments.
 *          The returned value doesn't represent whether or not log was output to console or file.
 */
- (BOOL)logWithMask:(PPLogControllerMask)mask format:(NSString *)format arguments:(va_list)arguments;

/*!
 *  Setting file name, & at Path, note file name & atPath can only be assigned once.
 *  if filename was not successfully set, warning message will appear in console in DEBUG mode.
 *
 *  @param fileName Name of the log file
 *  @param atPath   path of file
 **/
- (void)setFileName:(NSString *)fileName atPath:(NSString *)atPath;

/*!
 *  關閉目前的log檔，並建立新的log檔。（避免單一log檔過大）
 **/
- (void)renewLogFile;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Log Cost Time Methods

/**
 * 開始記錄花費時間
 * @param logKey 用這個來區分是否相同的log
 */
- (void)startLogCostTime:(NSString *)logKey;

/**
 * 顯示與上次log到現在花費的時間
 * @param logKey 用這個來區分是否相同的log, 需與startLogCostTime傳入的一致
 *  @param mask      Mask of log level.
 *  @param format    Format of string.
 *  arguments One or more arguments to be inserted into the string.
 *
 *  @return Return YES if log message is successfully composed by format and arguments.
 *          The returned value doesn't represent whether or not log was output to console or file.
 */
- (BOOL)logWithKey:(NSString *)logKey mask:(PPLogControllerMask)mask format:(NSString *)format, ...;

/**
 * 停止記錄花費時間，一定要call不然可能會有leak, 或時間計算問題
 * @param logKey 用這個來區分是否相同的log
 */
- (void)stopLogCostTime:(NSString *)logKey;



////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Methods

/*!
 *  Return a singleton instance of PPLogController.
 *
 *  @return return singleton of PPLogController.
 */
+ (instancetype)shareLogController;

/*!
 *  Write NsData to specific file name under self.filePath,
 *  you need to call setFileName:atPath: before calling this method.
 *
 *  Note: use PPLogData() macro instead of calling this method directly,
 *        unless you specifically intend to do so.
 *
 *  @param mask     Mask of log level.
 *  @param dataName Name of the file to save data with.
 *  @param data     The data to save.
 *
 *  @return Return YES if successfully write data to file.
 */
+ (BOOL)logWithMask:(PPLogControllerMask)mask dataName:(NSString *)dataName data:(NSData *)data;

/*!
 *  Output log with arguments.
 *
 *  Note: use PPLog() macro instead of calling this method directly,
 *        unless you specifically intend to do so.
 *
 *  @param mask      Mask of log level.
 *  @param format    Format of string.
 *  arguments One or more arguments to be inserted into the string.
 *
 *  @return Return YES if log message is successfully composed by format and arguments.
 *          The returned value doesn't represent whether or not log was output to console or file.
 */
+ (BOOL)logWithMask:(PPLogControllerMask)mask format:(NSString *)format, ...;



////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - (Class) Log cost time

/**
 * 開始記錄花費時間
 * @param logKey 用這個來區分是否相同的log
 */
+ (void)startLogCostTime:(NSString *)logKey;

/**
 * 顯示與上次log到現在花費的時間
 * @param logKey 用這個來區分是否相同的log, 需與startLogCostTime傳入的一致
 *  @param mask      Mask of log level.
 *  @param format    Format of string.
 *  arguments One or more arguments to be inserted into the string.
 *
 *  @return Return YES if log message is successfully composed by format and arguments.
 *          The returned value doesn't represent whether or not log was output to console or file.
 */
+ (BOOL)logWithKey:(NSString *)logKey mask:(PPLogControllerMask)mask format:(NSString *)format, ...;

/**
 * 停止記錄花費時間，一定要call不然可能會有leak, 或時間計算問題
 * @param logKey 用這個來區分是否相同的log
 */
+ (void)stopLogCostTime:(NSString *)logKey;

@end
