//
//  WCGroupViewDataSource.h
//  ADAL
//
//  Created by sanhue on 2018/2/22.
//

#import <Foundation/Foundation.h>

@class WCGroupView;

@protocol WCGroupViewDataSource <NSObject>

/**
 *  Ask the data source to return the number of child items encompassed by a given item. (required)
 *
 *  @param groupView     The tree-view that sent the message.
 *  @param item         An item identifying a cell in tree view.
 *
 *  @return The number of child items encompassed by item. If item is nil, this method should return the number of children for the top-level item.
 */
- (NSInteger)groupView:(nonnull WCGroupView *)groupView numberOfChildrenOfItem:(nullable id)item;

/**
 *  Ask the data source to return the child item at the specified index of a given item. (required)
 *
 *  @param groupView The tree-view object requesting child of the item at the specified index.
 *  @param index    The index of the child item from item to return.
 *  @param item     An item identifying a cell in tree view.
 *
 *  @return The child item at index of a item. If item is nil, returns the appropriate child item of the root object.
 */
- (nonnull id)groupView:(nonnull WCGroupView *)groupView child:(NSInteger)index ofItem:(nullable id)item;


@optional

///------------------------------------------------
/// Editing Tree Rows
///------------------------------------------------

/**
 *  Asks the data source to commit the insertion or deletion of a row for specified item in the receiver.
 *
 *  @param groupView     The tree-view object requesting the insertion or deletion.
 *  @param editingStyle The cell editing style corresponding to a insertion or deletion requested for the row specified by item. Possible editing styles are `UITableViewCellEditingStyleInsert` or `UITableViewCellEditingStyleDelete`.
 *  @param item         An item identifying a cell in tree view.
 */
- (void)groupView:(nonnull WCGroupView *)groupView commitEditingStyle:(UITableViewCellEditingStyle)editingStyle forRowForItem:(nullable id)item;

/**
 *  Asks the data source to verify that row for given item is editable.
 *
 *  @param groupView     The tree-view object requesting this information.
 *  @param item         An item identifying a cell in tree view.
 *
 *  @return `YES` if the row indicated by indexPath is editable; otherwise, `NO`.
 */
- (BOOL)groupView:(nonnull WCGroupView *)groupView canEditRowForItem:(nullable id)item;

/**
 *  Asks the delegate for the height to use for a row for a specified item.
 *
 *  @param groupView     The tree-view object requesting this information.
 *  @param item         An item identifying a cell in tree view.
 *
 *  @return A nonnegative floating-point value that specifies the height (in points) that row should be.
 */
- (CGFloat)groupView:(nonnull WCGroupView *)groupView  heightForRowForItem:(nullable id)item;
@end
