//
//  PPScrollTabController.h
//  Example
//
//  Created by sanhue on 2015/6/9.
//  Copyright (c) 2015年 Eddie. All rights reserved.
//

#import <UIKit/UIKit.h>
#import "PPViewController.h"
#import "PPButton.h"

////////////////////////////////////////////////////////////////////////////////////////////////////
@class PPScrollTabController;

@protocol PPScrollTabControllerDelegate <NSObject>

@optional
/// scrollTabController處理完view will appear後會收到
- (void)scrollTabController:(PPScrollTabController *)scrollTabController willAppear:(BOOL)animated;

/// scrollTabController處理完view did appear後會收到
- (void)scrollTabController:(PPScrollTabController *)scrollTabController didAppear:(BOOL)animated;

/// scrollTabController處理完view will disappear後會收到
- (void)scrollTabController:(PPScrollTabController *)scrollTabController willDisappear:(BOOL)animated;

/// scrollTabController處理完view did disappear後會收到
- (void)scrollTabController:(PPScrollTabController *)scrollTabController didDisappear:(BOOL)animated;

/**
 * 將要顯示哪一個頁面
 * @param scrollTabController 委託者
 * @param index 將要顯示的頁面index
 */
- (void)scrollTabController:(PPScrollTabController *)scrollTabController willShowPageAtIndex:(NSInteger)index;

/**
 * 已經顯示到某個頁面
 * @param scrollTabController 委託者
 * @param index 將要顯示的頁面index
 */
- (void)scrollTabController:(PPScrollTabController *)scrollTabController didShowPageAtIndex:(NSInteger)index;

@end




////////////////////////////////////////////////////////////////////////////////////////////////////
@protocol PPScrollTabControllerDataSource <NSObject>

/// 回傳tab個數
- (NSInteger)numberOfTabAtController:(PPScrollTabController *)scrollTabController;

/// 依據index回傳該頁的viewController
- (UIViewController *)scrollTabController:(PPScrollTabController *)scrollTabController viewControllerAtIndex:(NSInteger)index;

@optional

/// 依據index回傳tab button上的image
- (UIImage *)scrollTabController:(PPScrollTabController *)scrollTabController tabImageAtIndex:(NSInteger)index controlState:(UIControlState)controlState;

/// 依據index回傳tab button上的title
- (NSString *)scrollTabController:(PPScrollTabController *)scrollTabController tabTitleAtIndex:(NSInteger)index;

/// 依據index回傳tab button, 用來客制化tab button, 如果有實作此delegate, 就不會呼叫|scrollTabController:tabImageAtIndex:|,|scrollTabController:tabTitleAtIndex:|
- (UIButton *)scrollTabController:(PPScrollTabController *)scrollTabController buttonAtIndex:(NSInteger)index;
@end



////////////////////////////////////////////////////////////////////////////////////////////////////

/**
 * 一個可包含多個view controller的元件，可用左右的手勢切換頁面。
 * 上方包含由多個按鈕組成的功能列，按鈕數量與view controller的數量一致，
 * 按鈕名稱自動抓取view controller的title
 */
@interface PPScrollTabController : PPViewController <PPButtonDelegate, UIScrollViewDelegate>

/// delegate
@property (nonatomic, assign) id<PPScrollTabControllerDelegate> delegate;
/// datasource
@property (nonatomic, assign) id<PPScrollTabControllerDataSource> dataSource;

/// 元件中所包含的所有view controller，
@property (nonatomic, retain, readonly) NSArray *viewControllers;

/// 目前顯示的view controller，
@property (nonatomic, retain, readonly) UIViewController *selectedViewController;

/// 目前元件所顯示的頁面的index
@property (nonatomic, assign) NSInteger currentTabIndex;

/// 是否能夠scroll, default YES，不能scroll時會同時鎖住上方的tab button
@property (nonatomic, assign) BOOL enableScroll;



////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - style

/// 設定tab button background color, default white
@property (nonatomic, retain) UIColor *tabButtonColor;

/// 設定tab button title color, default black
@property (nonatomic, retain) UIColor *tabButtonTitleColor;

/// 設定tab button selected title color, default blue
@property (nonatomic, retain) UIColor *tabButtonTitleSelectedColor;

/// 設定tab button image edge insets, default UIEdgeInsetsZero
@property (nonatomic, assign) UIEdgeInsets buttonImageEdgeInsets;

/// 設定tab button image minimum height, default 0, 不限制
@property (nonatomic, assign) CGFloat buttonImageMinimumHeight;

/// 設定tab indicator color, default white
@property (nonatomic, retain) UIColor *tabIndicatorColor;

/// 設定tab indicator color, default white
@property (nonatomic, retain) UIColor *tabIndicatorBackgroundColor;

/// 設定tab indicator height, default 3
@property (nonatomic, assign) CGFloat tabIndicatorHeight;

/// 設定下面工具列高度，Scroll view只大小不會超過bottom toolbar, default 0
@property (nonatomic, assign) CGFloat bottomToolbarHeight;

/// layout時是否考慮bottomLayoutGuide, default NO
@property (nonatomic, assign) BOOL layoutByBottomLayoutGuide;

//////////////////////////////////////////////////

/**
 * 設定scroll手勢在gestureRecognizer失敗之後才生效
 * @param gestureRecognizer 優先執行的手勢
 */
- (void)pageScrollViewSetRequireGestureRecognizerToFail:(UIGestureRecognizer *)gestureRecognizer;

/**
 * 取得tab button instance
 * @param index index of tab button 
 * @return PPButton tab button at index
 */
- (PPButton *)tabButtonWithIndex:(NSInteger)index;
@end
