//
//  PPCameraMedia.h
//
//  Created by pptai on 13/4/3.
//  Copyright (c) 2013年 penpower. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <AVFoundation/AVFoundation.h>

#if TARGET_OS_IPHONE
#import <UIKit/UIKit.h>
#endif

#import "CrossPlatformDefine.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

/**
 相機型態

 - PPCameraType_Front: 前鏡頭
 - PPCameraType_Back: 後鏡頭
 */
typedef NS_ENUM(NSUInteger, PPCameraType)
{
    PPCameraType_Front,
    PPCameraType_Back,
};


typedef NS_ENUM(NSUInteger, PPCameraHDRMode)
{
    PPCameraHDRMode_Auto,
    PPCameraHDRMode_On,
	PPCameraHDRMode_Off,
};


extern NSString *const PPCameraView_focus;
////////////////////////////////////////////////////////////////////////////////////////////////////

@class PPCameraView;

@protocol PPCameraViewDelegate <NSObject>

@optional
 // 使用captureStillImagWithShackingCheck:帶入YES時，在等待過程中會回傳震動程度。
 // shakingLevel：數值越大代表越不穩定
- (void)ppCameraView:(PPCameraView *)ppCameraView shakingLevel:(CGFloat)shakingLevel;
- (void)ppCameraView:(PPCameraView *)ppCameraView didReceivePreviewImage:(CPImage *)image;
- (void)ppCameraView:(PPCameraView *)ppCameraView didGetStillImage:(CPImage *)image;

// 收到qrcode
- (void)ppCameraView:(PPCameraView *)ppCameraView didGetQrcode:(NSString *)qrcode;
// 桌面錄影結束
- (void)ppCameraview:(PPCameraView *)ppCameraView didFinishRecordWithURL:(NSURL *)url error:(NSError *)error;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPCameraMedia

@interface PPCameraView : CPView

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Common Property

/// delegate
@property (nonatomic,assign) id<PPCameraViewDelegate> delegate;

/// sessionPreset, default AVCaptureSessionPresetPhoto
/// 需在 |startCameraPreview| 前設定
@property (nonatomic,retain) NSString *sessionPreset;

/// 設定取得的still image的最大邊的大小
/// 預設為0，表示不調整，如果不為0，且image size大於設定值，會將輸出的影像調整大小後輸出
@property (nonatomic,assign) CGFloat maxStillImageLength;

/// 是否依照preview畫面比例調整輸出影像，預設為YES，需要直出原圖時請設為NO。
@property (nonatomic,assign) BOOL needCropImageByPreviewSize;

/// 目前相機使用前鏡頭或是後鏡頭，預設為 PPCameraType_Back
@property (nonatomic,assign) PPCameraType  ppCameraType;

/// 是否指定鏡頭名稱
@property (nonatomic,retain) NSString *cameraDeviceName;

// 是否支援鏡像輸出
@property (nonatomic,assign) BOOL supportVideoMirrored;

/// 取得目前的capture sesion
@property (atomic,retain) AVCaptureSession           *session;

@property (atomic,retain) AVCaptureMovieFileOutput *captureMovieFileOutput;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - IOS Property

#if TARGET_OS_IPHONE


/// 是否調整輸出previewImage的方向，會影響到的是|ppCameraView:didReceivePreviewImage:|取得的image，預設為 NO
@property (nonatomic,assign) BOOL adjustOutputImageOrientation;

/// 是否允許點選畫面做focus，預設為 YES
@property (nonatomic,assign) BOOL enableFocusTouch;

/// 設定目前介面方向 (super view旋轉時一定要呼叫！！)
/// adjustOutputImageOrientation==YES時會以此方向來旋轉圖片，預設為 UIInterfaceOrientationPortrait
@property (nonatomic,assign) UIInterfaceOrientation interfaceOrientation;

/// 如果cameraView的parent view是不轉的，要設為YES，如果cameraView的parent view會旋轉，要設為NO, 預設NO;
@property (nonatomic,assign) BOOL viewOrientationFix;

/// 設定focus時顯示的圖示，預設使用PPCameraView_focus.png的圖檔為focus image, 可更換此圖檔，或在此指定要用的focus image
@property (nonatomic,retain) UIImage *focusImage;

/// 設定自動對焦的模式，預設為 AVCaptureFocusModeContinuousAutoFocus
@property (nonatomic, assign) AVCaptureFocusMode focusMode;

/// 設定曝光的模式，預設為 AVCaptureExposureModeContinuousAutoExposure
@property (nonatomic, assign) AVCaptureExposureMode exposureMode;

/// 設定白平衡的模式，預設為 AVCaptureWhiteBalanceModeContinuousAutoWhiteBalance
@property (nonatomic, assign) AVCaptureWhiteBalanceMode whiteBalanceMode;

/// 設定縮放倍率
@property (nonatomic, assign) CGFloat videoZoomFactor;

@property(nonatomic, readonly) AVCaptureDeviceType deviceType;

/// 設定對焦點，使用view的座標系，不是device的座標系。
@property (nonatomic, assign) CGPoint focusPoint;

/// 只要有設定對焦點就顯示對焦影像，預設為NO。（Tap一定會顯示不被此參數影響）
@property (nonatomic, assign) BOOL alwaysShowFocusImageWhenSetFoucsPoint;

/// preview 穩定度偵測，開啟時，偵測到穩定才會送出 |ppCameraView:didReceivePreviewImage:|
@property (nonatomic, assign, getter=isEnablePreviewStableCheck) BOOL enablePreviewStableCheck;

#endif


////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance control Method

/**
 * 設定相機並開始preview
 */
- (BOOL)startCameraPreview;
- (BOOL)startCameraPreviewWithCompletion:(void(^)(void))completion;
/**
 * 停止preview並將相機資源釋放
 */
- (void)stopCameraPreview;

/**
 * 暫停preview
 */
- (void)pauseCameraPreview;

/**
 * 回復preview
 */
- (void)resumeCameraPreview;






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Video Input Method

- (BOOL)addVideoInputCamera;






////////////////////////////////////////////////////////////////////////////////////////////////////

#if TARGET_OS_IPHONE

#pragma mark - IOS Instance Method


/// 相機是否支援HDR
- (BOOL)isSupportHDR;

/// 設定相機HDR模式
/// @param hdrMode HDR模式
- (void)setHDRMode:(PPCameraHDRMode)hdrMode;

/**
 * 前後鏡頭切換
 */
- (void)switchCameras;

/**
 * 開關閃光燈
 * @param turnOn YES: 開啟閃光燈，NO: 不開閃光燈
 * @param useAutoMode turnOn為YES時才有做用，YES-自動判斷是否需打閃光燈，NO-每次都打閃光燈
 */
- (void)turnOnFlash:(BOOL)turnOn useAutoMode:(BOOL)useAutoMode;

/**
 * 開關手電筒 (閃光燈恆亮)
 * @param turnOn YES: 開啟手電筒，NO: 關閉手電筒
 */
- (void)turnOnTorch:(BOOL)turnOn;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - 超廣角 Method (IOS)

/// 相機是否支援超廣角
- (BOOL)isSupportUltraWideCamera;

/// 使用超廣角
- (void)ultraWideCameraEnable:(BOOL)enable;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance focus Method (IOS)

/**
 * 查詢是否正在做focus
 * @return BOOL YES: 正在focus, NO: 沒有在做focus
 */
- (BOOL)isFocusing;





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance Qrcode Method (IOS)
/**
 * 開啟qrcode辨讀
 * @param enable YES: 開啟，NO: 關閉
 */
- (void)enableQrcodeRecognize:(BOOL)enable;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance layout Method (IOS)

/**
 * 取得目前的preview layer
 * @param view superView
 * @return AVCaptureVideoPreviewLayer 目前在做preview 的layer
 */
- (AVCaptureVideoPreviewLayer *)previewInView:(CPView *)view;

/**
 * 重設preview大小
 * call |layoutPreviewInView:withDefaultOrientation:|, orientation的值為UIInterfaceOrientationPortrait
 * 需在 |startCameraPreview| call才能正確設定videoOrientation
 * @param view preview所在的view
 */
- (void)layoutPreviewInView:(CPView *)view;

/**
 * 重設preview大小，並以指定的方向來layout preview
 * 需在 |startCameraPreview| call才能正確設定videoOrientation
 * @param view preview所在的view
 * @param orientation 指定preview layer的方向 (一般來說只要傳UIInterfaceOrientationPortrait)就好
 */
- (void)layoutPreviewInView:(CPView *)view withDefaultOrientation:(UIInterfaceOrientation)orientation;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - capture Method (IOS)

/**
 * 拍照 (影像透過delegate的|ppCameraView:didGetStillImage:|回傳)
 * @param needStableCheck 是否進行震動檢查（震動程度透過delegate的|ppCameraView:shakingLevel:|回傳）
 */
- (void)captureStillImagWithStableCheck:(BOOL)needStableCheck;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instace Method (macOS)

#elif TARGET_OS_MAC


/**
 * 拍照 (影像透過delegate的|ppCameraView:didGetStillImage:|回傳)
 */
- (void)captureStillImage;

#endif





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Method

/**
 * 取得相機數量
 * @return NSInteger 相機數量
 */
+ (NSInteger)numberOfCameras;

/**
 * 取得背面相機是否可用
 * @return BOOL YES: 可用；NO: 不可用
 */
+ (BOOL)backCameraAvailable;

/**
 * 取得正面相機是否可用
 * @return BOOL YES: 可用；NO: 不可用
 */
+ (BOOL)frontCameraAvailable;

/**
 * 取得背面相機的CaptureDevice實體
 * @return AVCaptureDevice 如果不能使用回傳nil
 */
+ (AVCaptureDevice *)backCamera;

/**
 * 取得背面相機的CaptureDevice實體
 * @return AVCaptureDevice 如果不能使用回傳nil
 */
+ (AVCaptureDevice *)frontCamera;

/**
 * 依cameraType判斷是否可使用閃光燈
 * @param cameraType 正面或反面相機
 * @return BOOL YES: 可用；NO: 不可用
 */
+ (BOOL)canUseFlashForCameraType:(PPCameraType)cameraType;

/**
 * 依cameraType判斷是否可使用手電筒
 * @param cameraType 正面或反面相機
 * @return BOOL YES: 可用；NO: 不可用
 */
+ (BOOL)canUseTorchForCameraType:(PPCameraType)cameraType;

/**
 * 取得相機列表
 * @param position 鏡頭位置，macOS目前無法指定，都用AVCaptureDevicePositionUnspecified
 * @return NSArray  array of AVCaptureDevice
 */
+ (NSArray *)captureDeviceWithPosition:(AVCaptureDevicePosition)position;

@end
