//
//  NSError+Custom.m
//  
//
//  Created by Mike on 13/4/26.
//  Modified by Egg on 2015/01/22.
//

#import "NSError+Custom.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

NSString * const NSErrorCustom_Key_LineNumber  = @"NSErrorCustom_Key_LineNumber";
NSString * const NSErrorCustom_Key_Object      = @"NSErrorCustom_Key_Object";

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation NSError (Custom)





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Class Methods

//================================================================================
//
//================================================================================
+ (NSDictionary *)dictionaryWithLocalizedFailureReason:(NSString *)localizedFailureReason
                                            lineNumber:(NSUInteger)lineNumber
                                                object:(id)object
{
    if (localizedFailureReason == nil)
    {
        localizedFailureReason = @"";
    }
    
    if (object == nil)
    {
        return @{ NSErrorCustom_Key_LineNumber : [NSString stringWithFormat:@"%lu", (unsigned long)lineNumber],
                  NSLocalizedFailureReasonErrorKey: localizedFailureReason
                  };
    }
    return @{ NSErrorCustom_Key_LineNumber : [NSString stringWithFormat:@"%lu", (unsigned long)lineNumber],
              NSLocalizedFailureReasonErrorKey: localizedFailureReason,
              NSErrorCustom_Key_Object : object
              };
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Public Class Methods

//================================================================================
//
//================================================================================
+ (NSError *)errorWithDomain:(NSString *)domain
                        code:(NSInteger)code
                  lineNumber:(NSUInteger)lineNumber
      localizedFailureReason:(NSString *)localizedFailureReason
                      object:(id)object
{
    return [NSError errorWithDomain:domain
                               code:code
                           userInfo:[self dictionaryWithLocalizedFailureReason:localizedFailureReason
                                                                    lineNumber:lineNumber
                                                                        object:object]];
}


//================================================================================
//
//================================================================================
+ (BOOL)makeError:(NSError **)error
       withDomain:(NSString *)domain
             code:(NSInteger)code
       lineNumber:(NSUInteger)lineNumber
           object:(id)object
    checkHasError:(BOOL)checkHasError
   throwWhenDebug:(BOOL)throwWhenDebug
           format:(NSString *)format, ...
{

    //////////////////////////////////////////////////
    // Make error message
    NSString *errorMessage = nil;
    va_list arguments;
    va_start(arguments, format);
    errorMessage = [[[NSString alloc] initWithFormat:format arguments:arguments] autorelease];
    va_end(arguments);
    
    //////////////////////////////////////////////////
    // Make userInfo
    NSDictionary *userInfo = [self dictionaryWithLocalizedFailureReason:errorMessage
                                                             lineNumber:lineNumber
                                                                 object:object];

    //////////////////////////////////////////////////
    // Throw exception if DEBUG && throwWhenDebug
    if (throwWhenDebug == YES)
    {
#if DEBUG
        @throw [NSException exceptionWithName:domain
                                       reason:errorMessage
                                     userInfo:userInfo];
#endif
    }
    
    //////////////////////////////////////////////////
    // Check for conditions
    if(error == nil)
    {
        return NO;
    }

    if (checkHasError == YES && *error != nil)
    {
        return NO;
    }
    
    //////////////////////////////////////////////////
    // Make error
    *error = [NSError errorWithDomain:domain
                                 code:code
                             userInfo:userInfo];
   
    return YES;
}



////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Methods

//================================================================================
//
//================================================================================
- (NSString *)alertMessage
{
    if([self.localizedFailureReason length] > 0 ||
       [self.localizedDescription length] > 0)
    {
        NSMutableString *message = [NSMutableString string];
        
//        [message appendFormat:@"[%@:%ld]\n", self.domain, (long)self.code];
        
        if([self.localizedFailureReason length] > 0)
        {
            [message appendFormat:@"%@", self.localizedFailureReason];
        }
        else
        {
            [message appendFormat:@"%@", self.localizedDescription];
        }
        
        return message;
    }
    
    return self.description;
}


//================================================================================
//
//================================================================================
- (NSError *)findErrorWithConditionDict:(NSDictionary *)condictionDict
{
    NSError *returnError = nil;
    NSArray *codes = [condictionDict objectForKey:self.domain];
    
    if([codes containsObject:@(self.code)])
    {
        returnError = self;
    }
    else if(self.userInfo != nil)
    {
        NSArray *infoValues = [self.userInfo allValues];
        
        for(id value in infoValues)
        {
            if([value isKindOfClass:[NSError class]])
            {
                NSError *subError = (NSError *)value;
                
                returnError = [subError findErrorWithConditionDict:condictionDict];
                
                if(returnError != nil)
                {
                    break;
                }
            }
        }
        
    }
    
    return returnError;
}


//================================================================================
//
//================================================================================
- (NSError *)findNetworkError
{
    NSDictionary *condictionDict = @{@"NSURLErrorDomain" : @[@(NSURLErrorTimedOut), // -1001
                                                             @(NSURLErrorCannotConnectToHost), //-1004
                                                             @(NSURLErrorNetworkConnectionLost), // -1005
                                                             @(NSURLErrorNotConnectedToInternet), // -1009
                                                             @(NSURLErrorSecureConnectionFailed)], // -1200
                                     @"kCFErrorDomainCFNetwork" : @[@(kCFURLErrorTimedOut), // -1001
                                                                    @(kCFURLErrorCannotConnectToHost), //-1004
                                                                    @(kCFURLErrorNetworkConnectionLost), // -1005
                                                                    @(kCFURLErrorNotConnectedToInternet)], // -1009
                                     @"com.google.HTTPStatus" : @[@(-1001),
                                                                  @(-1004),
                                                                  @(-1005),
                                                                  @(-1009)]};
    
    return [self findErrorWithConditionDict:condictionDict];
}


//================================================================================
//
//================================================================================
- (NSError *)findStorageError
{
    NSDictionary *condictionDict = @{@"WCCardDBController" : @[@(13)],  // database or disk is full
                                     @"WCDataController" : @[@(8001)]   // WCDataController_Error_NotEnoughDiskSpace
                                     };
    
    return [self findErrorWithConditionDict:condictionDict];
}


//================================================================================
//
//================================================================================
- (NSString *)googleDescription
{
    NSString *description = @"";
    NSData *data = [self.userInfo objectForKey:@"data"];
    
    if(data != nil && [self.domain isEqualToString:@"com.google.HTTPStatus"] == YES)
    {
        NSString *dataString = [[NSString alloc] initWithData:data encoding:NSUTF8StringEncoding];
        
        description = [NSString stringWithFormat:@"Error Domain=%@ Code=%ld UserInfo=\"%@\"", self.domain, (long)self.code, dataString];
        [dataString release];
    }
    else
    {
        description = [self description];
    }
    
    return description;
}


//================================================================================
//
//================================================================================
- (NSString *)detailDescription
{
    NSMutableString *description = [NSMutableString stringWithFormat:@"{Error Domain=%@ Code=%ld \"%@\"", self.domain, (long)self.code, self.localizedFailureReason];
    
    if(self.userInfo != nil)
    {
        NSArray *infoKeys = [self.userInfo allKeys];
        NSArray *infoValues = [self.userInfo allValues];
        
        [description appendString:@" UserInfo={"];
        
        for(int i=0; i<[infoKeys count]; i++)
        {
            NSString *key = infoKeys[i];
            id value = infoValues[i];
            
            // 特殊處理的key
            if([key isEqualToString:@"data"] == YES) // com.google.HTTPStatus
            {
                NSData *data = (NSData *)value;
                NSString *dataString = [[[NSString alloc] initWithData:data encoding:NSUTF8StringEncoding] autorelease];
                
                if([dataString length] > 0)
                {
                    value = dataString;
                }
            }
            else if([key isEqualToString:NSErrorCustom_Key_LineNumber] == YES)
            {
                key = @"Line";
            }
            else if([key isEqualToString:NSLocalizedFailureReasonErrorKey] == YES)
            {
                key = @"Reason";
            }
            else
            {
                if([key isEqualToString:NSErrorCustom_Key_Object] == YES)
                {
                    key = @"SubError";
                }
                
                if([value isKindOfClass:[NSError class]])
                {
                    NSError *subError = (NSError *)value;
                    value = [subError googleDescription];
                }
            }
            
            if(i>0)
            {
                [description appendString:@", "];
            }
            
            [description appendFormat:@"%@=%@", key, value];
        }
        
        [description appendString:@"}}"];
    }
    
    return description;
}

@end
