//
//  NSData+Base64.m
//  
//
//  Created by Mike on 13/8/13.
//  
//

#import "NSData+Base64.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#define PPBase64DecodingTableSize 128
#define PPBase64EncodingTableSize 64

////////////////////////////////////////////////////////////////////////////////////////////////////

static char base64EncodingTable_Standard[]  = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
static char base64EncodingTable_WebSafe[]   = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_";

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation NSData (Base64)

//================================================================================
//
//================================================================================
- (NSString *)base64EncodedStringWithTable:(const char *)table
{
    NSString *base64EncodedString = nil;
    
    if(table!=NULL)
    {
        NSUInteger  length      = [self length];
        NSUInteger  bufferSize  = ((length+2)/3)*4;
        
        //////////////////////////////////////////////////
        
        Byte *buffer = (Byte *)malloc(bufferSize);
        if(buffer!=NULL)
        {
            const Byte *bytes = [self bytes];
            
            for(NSUInteger bytesIndex=0; bytesIndex<length; bytesIndex+=3)
            {
                NSUInteger value = 0;
                
                for(NSUInteger j=bytesIndex; j<(bytesIndex+3); j++)
                {
                    value <<= 8;
                    
                    if(j<length)
                    {
                        value |= (0xFF & bytes[j]);
                    }
                }
                
                NSInteger bufferIndex = (bytesIndex/3)*4;
                
                buffer[bufferIndex + 0] =                           table[(value >> 18) & 0x3F];
                buffer[bufferIndex + 1] =                           table[(value >> 12) & 0x3F];
                buffer[bufferIndex + 2] = (bytesIndex+1)<length ?   table[(value >>  6) & 0x3F] : '=';
                buffer[bufferIndex + 3] = (bytesIndex+2)<length ?   table[(value >>  0) & 0x3F] : '=';
            }
            
            //////////////////////////////////////////////////
            
            base64EncodedString = [[[NSString alloc] initWithBytes:buffer
                                                            length:bufferSize
                                                          encoding:NSASCIIStringEncoding] autorelease];
            
            //////////////////////////////////////////////////
            
            free(buffer);
        }
    }
    
    return base64EncodedString;
}

//================================================================================
//
//================================================================================
- (NSString *)base64EncodedStandardString
{
    return [self base64EncodedStringWithTable:base64EncodingTable_Standard];
}

//================================================================================
//
//================================================================================
- (NSString *)base64EncodedWebSafeString
{
    return [self base64EncodedStringWithTable:base64EncodingTable_WebSafe];
}

//================================================================================
//
//================================================================================
+ (void)convertBase64EncodingTable:(const char *)encodingTable toBase64DecodingTable:(char *)decodingTable
{
    memset(decodingTable, 0, PPBase64DecodingTableSize);
    
    for(NSUInteger index=0; index<PPBase64EncodingTableSize; index++)
    {
        decodingTable[(unsigned int) encodingTable[index]] = (char)index;
    }
}

//================================================================================
//
//================================================================================
+ (NSData *)dataWithBase64EncodedString:(NSString *)string table:(const char *)table
{
    NSData *data = nil;
    
    do
    {
        if(string==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        const char *cString = [string cStringUsingEncoding:NSASCIIStringEncoding];
        if(cString==NULL)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSInteger cStringLength = strlen(cString);
        if((cStringLength%4)!=0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(cStringLength==0)
        {
            data = [NSData data];
            break;
        }
        
        //////////////////////////////////////////////////

        while(cStringLength>0 && cString[cStringLength-1]=='=')
        {
            cStringLength--;
        }
        
        //////////////////////////////////////////////////
        
        NSUInteger  bufferSize = cStringLength*3/4;
        
        Byte *buffer = (Byte *)malloc(bufferSize);
        if(buffer!=NULL)
        {
            NSInteger cStringIndex  = 0;
            NSInteger bufferIndex   = 0;
            
            while(cStringIndex<cStringLength)
            {
                int i0 = cString[cStringIndex++];
                int i1 = cString[cStringIndex++];
                int i2 = cStringIndex<cStringLength ? cString[cStringIndex++] : 'A'; // 'A' will decode to \0
                int i3 = cStringIndex<cStringLength ? cString[cStringIndex++] : 'A';
                
                buffer[bufferIndex++] = (uint8_t)((table[i0] << 2) | (table[i1] >> 4));
                
                if(bufferIndex < bufferSize)
                {
                    buffer[bufferIndex++] = (uint8_t)(((table[i1] & 0xF) << 4) | (table[i2] >> 2));
                }
                
                if(bufferIndex < bufferSize)
                {
                    buffer[bufferIndex++] = (uint8_t)(((table[i2] & 0x3) << 6) | table[i3]);
                }
            }
            
            //////////////////////////////////////////////////
            
            data = [NSData dataWithBytes:buffer length:bufferSize];
            
            //////////////////////////////////////////////////
            
            free(buffer);
        }
        
    }while(0);
    
    return data;
}

//================================================================================
//
//================================================================================
+ (NSData *)dataWithBase64EncodedStandardString:(NSString *)string
{
    static char base64DecodingTable_Standard[PPBase64DecodingTableSize];
    static BOOL converted = NO;
    
    if(converted==NO)
    {
        [NSData convertBase64EncodingTable:base64EncodingTable_Standard toBase64DecodingTable:base64DecodingTable_Standard];
        converted = YES;
    }
    
    return [NSData dataWithBase64EncodedString:string table:base64DecodingTable_Standard];
}

//================================================================================
//
//================================================================================
+ (NSData *)dataWithBase64EncodedWebSafeString:(NSString *)string
{
    static char base64DecodingTable_WebSafe[PPBase64DecodingTableSize];
    static BOOL converted = NO;
    
    if(converted==NO)
    {
        [NSData convertBase64EncodingTable:base64EncodingTable_WebSafe toBase64DecodingTable:base64DecodingTable_WebSafe];
        converted = YES;
    }
    
    return [NSData dataWithBase64EncodedString:string table:base64DecodingTable_WebSafe];
}

@end
