//
//  NSData+AES256.m
//  
//
//  Created by Mike on 13/4/25.
//  
//

#import "NSData+AES256.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

NSData *shareInitializationVector_ =nil;

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation NSData (AES256)

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Methods

//================================================================================
//
//================================================================================
- (NSUInteger)keysizeForKey:(NSString *)key
{
    NSUInteger length = kCCKeySizeAES256;
    
    switch (key.length)
    {
        case 8:
        {
            length = kCCKeySizeAES128;
            break;
        }
        case 24:
        {
            length = kCCKeySizeAES192;
            break;
        }
        default:
        {
            break;
        }
    }
    
    return length;
}


//================================================================================
//
//================================================================================
- (NSData *)cryptDataWithOperation:(CCOptions)operation
                               key:(NSString *)key
                           keySize:(NSUInteger)keySize
                           options:(CCOptions)options
              initializationVector:(NSData *)initializationVector
{
    NSData *data =nil;
    
    do
    {
        // Setup Key
        Byte byteKey[keySize];
        bzero(byteKey, sizeof(byteKey));
        [[key dataUsingEncoding:NSUTF8StringEncoding] getBytes:byteKey length:sizeof(byteKey)];

        //////////////////////////////////////////////////
        // Setup Initialization Vector
        
        Byte byteInitializationVector[kCCBlockSizeAES128];
        bzero(byteInitializationVector, sizeof(byteInitializationVector));
        if(initializationVector!=NULL)
        {
            [initializationVector getBytes:byteInitializationVector length:sizeof(byteInitializationVector)];
        }
        
        //////////////////////////////////////////////////
        
        NSUInteger dataInLength = [self length];
        size_t dataOutAvailable = dataInLength+kCCBlockSizeAES128;
        
        //////////////////////////////////////////////////
        
        void *dataOut = malloc(dataOutAvailable);
        if(dataOut!=NULL)
        {
            size_t dataOutMoved = 0;
            CCCryptorStatus status = CCCrypt(operation,
                                             kCCAlgorithmAES128,
                                             options,
                                             byteKey,
                                             sizeof(byteKey),
                                             byteInitializationVector,
                                             [self bytes],
                                             dataInLength,
                                             dataOut,
                                             dataOutAvailable,
                                             &dataOutMoved);
            
            if(status==kCCSuccess)
            {
                data = [NSData dataWithBytesNoCopy:dataOut length:dataOutMoved];
            }
            else
            {
                free(dataOut);
            }
        }
        
    }while(0);
    
    return data;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Methods

//================================================================================
//
//================================================================================
- (NSData *)cryptDataWithOperation:(CCOptions)operation
                               key:(NSString *)key
              initializationVector:(NSData *)initializationVector
{
    return [self cryptDataWithOperation:operation
                                    key:key
                                keySize:kCCKeySizeAES256
                                options:kCCOptionPKCS7Padding
                   initializationVector:initializationVector];
}


//================================================================================
//
//================================================================================
- (NSData *)AES256EncryptDataWithKey:(NSString *)key initializationVector:(NSData *)initializationVector
{
    return [self cryptDataWithOperation:kCCEncrypt
                                    key:key
                                keySize:kCCKeySizeAES256
                                options:kCCOptionPKCS7Padding
                   initializationVector:initializationVector];
}

//================================================================================
//
//================================================================================
- (NSData *)AES256DecryptDataWithKey:(NSString *)key initializationVector:(NSData *)initializationVector
{
    return [self cryptDataWithOperation:kCCDecrypt
                                    key:key
                                keySize:kCCKeySizeAES256
                                options:kCCOptionPKCS7Padding
                   initializationVector:initializationVector];
}

//================================================================================
//
//================================================================================
- (NSData *)AES256EncryptDataWithKey:(NSString *)key
{
    return [self cryptDataWithOperation:kCCEncrypt
                                    key:key
                                keySize:kCCKeySizeAES256
                                options:kCCOptionPKCS7Padding
                   initializationVector:nil];
}

//================================================================================
//
//================================================================================
- (NSData *)AES256DecryptDataWithKey:(NSString *)key
{
    return [self cryptDataWithOperation:kCCDecrypt
                                    key:key
                                keySize:kCCKeySizeAES256
                                options:kCCOptionPKCS7Padding
                   initializationVector:nil];
}


//================================================================================
//
//================================================================================
- (NSData *)AESEncryptDataWithKey:(NSString *)key options:(CCOptions)options
{
    return [self cryptDataWithOperation:kCCEncrypt
                                    key:key
                                keySize:[self keysizeForKey:key]
                                options:options
                   initializationVector:nil];
}


//================================================================================
//
//================================================================================
- (NSData *)AESDecryptDataWithKey:(NSString *)key options:(CCOptions)options
{
    return [self cryptDataWithOperation:kCCDecrypt
                                    key:key
                                keySize:[self keysizeForKey:key]
                                options:options
                   initializationVector:nil];
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Methods

//================================================================================
//
//================================================================================
+ (NSData *)generateInitializationVector
{
    static dispatch_once_t onceToken;
    
    dispatch_once(&onceToken,^
    {
        srand((unsigned)time(NULL));
    });
    
    //////////////////////////////////////////////////
    
    char cInitializationVector[kCCBlockSizeAES128];
    
    for(int i=0; i<kCCBlockSizeAES128; i++)
    {
        cInitializationVector[i] = rand() % 256;
    }
    
    //////////////////////////////////////////////////
    
    return [NSData dataWithBytes:cInitializationVector length:kCCBlockSizeAES128];
}

//================================================================================
//
//================================================================================
+ (void)setShareInitializationVector:(NSData *)shareInitializationVector
{
    [shareInitializationVector retain];
    [shareInitializationVector_ release];
	shareInitializationVector_ = shareInitializationVector;
}

//================================================================================
//
//================================================================================
+ (NSData *)shareInitializationVector
{
    return shareInitializationVector_;
}

@end
